package model.firefighterscenario;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import javafx.scene.paint.Color;
import model.Board;
import model.Entity;
import model.Square;
import util.Direction;
import util.Position;
import util.PositionUtil;

public class AirTanker implements Entity{
    private final Color viewColor = Color.GOLD;
    private Direction direction; // Direction in which the AirTanker moves
    private int age;
    private Position position;
    private int priority = 3;

    public AirTanker(Position position, Board<Square> b, int age) {
        this.age = age;
        this.position = position;
        determineDirection(b);
        extinguish(position,b);
    }

    @Override
    public int getPriority() {
        return this.priority;
    }

    private void determineDirection(Board<Square> b) {
        int rowCount = b.rowCount();
        int columnCount = b.columnCount();

        if (position.x() == 0) {
            direction = Direction.EAST; // Move east if on left edge
        } else if (position.x() == columnCount - 1) {
            direction = Direction.WEST; // Move west if on right edge
        } else if (position.y() == 0) {
            direction = Direction.SOUTH; // Move south if on top edge
        } else if (position.y() == rowCount - 1) {
            direction = Direction.NORTH; // Move north if on bottom edge
        } else {
            // Not on an edge; default to moving east or choose a random direction
            direction = Direction.EAST;
        }
    }

    @Override
    public List<Position> nextTurn(Board<Square> b) {
        List<Position> positions = new ArrayList<>();
        // Move in the determined direction
        Position nextPos = getNextPosition(position, direction, b);
        
        if (nextPos == null || !b.doesPositionExist(nextPos)) {
            // Reached the edge; remove AirTanker from the board
            b.clearCaseFrom(this, position);
            positions.add(position);
            return positions;
        } else {
            // Extinguish fires within a 3-square radius

            positions.addAll(extinguish(position, b));
            // Move to the next position
            b.clearCaseFrom(this, position);
            positions.add(new Position(position.x(), position.y())); // Old position
            this.position = nextPos;
            b.addEntityAtSquare(this, nextPos);
            positions.add(nextPos); // New position

            return positions;
        }
    }
    protected List<Position> extinguish(Position position, Board<Square> b){
        List<Position> positions = new ArrayList<Position>();
        List<Position> positionsInRange = PositionUtil.getPositionsInRadius(position, 3, b);
        for (Position p : positionsInRange) {
            if (b.doesSquareContainEntity(p, Fire.class)) {
                b.getStates(p).getEntities().removeIf(e -> e instanceof Fire);
                positions.add(p); // Add position where fire was extinguished
            }
        }
        return positions;
    }
    private Position getNextPosition(Position currentPosition, Direction direction, Board<Square> b) {
        int x = currentPosition.x();
        int y = currentPosition.y();

        switch (direction) {
            case NORTH:
                y -= 1;
                break;
            case SOUTH:
                y += 1;
                break;
            case EAST:
                x += 1;
                break;
            case WEST:
                x -= 1;
                break;
            default:
                break; // Only handling cardinal directions
        }
        Position newPos = new Position(x, y);
        if (b.doesPositionExist(newPos)) {
            return newPos;
        } else {
            return null; // Reached the edge
        }
    }

    public Direction getDirection(){
        return this.direction;
    }

        @Override
    public void setPosition(Position p) {
        this.position = p;
    }

    @Override
    public Position getPosition() {
        return this.position;
    }

    @Override
    public Color getViewColor() {
        return this.viewColor;
    }

    @Override
    public int getAge() {
        return this.age;
    }

    @Override
    public void incrementAge() {
        this.age++;
    }

    @Override
    public void setAge(int age) {
        this.age = age;
    }



    @Override
    public boolean equals(Object obj) {
        if (this == obj) return true; // Check if same object
        if (obj == null || getClass() != obj.getClass()) return false; // Check for null and class match
        AirTanker airTanker = (AirTanker) obj; // Cast to Fire
        return age == airTanker.age && position.equals(airTanker.position); // Compare age and position
    }
    
    
    @Override
    public int hashCode() {
        return Objects.hash(position, age);
    }


}
