package model.firefighterscenario;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import javafx.scene.paint.Color;
import model.Board;
import model.Entity;
import model.Square;
import util.Position;
import util.PositionUtil;


public class Fire implements Entity {
    private Position position;
    private final Color viewColor = Color.RED;
    private int age;
    private final int priority = 1;

    public Fire(Position position) {
        this.position = position;
        this.age = 0;
    }

    public Fire(Position position, int age) {
        this.position = position;
        this.age = age;
    }

    @Override
    public List<Position> nextTurn(Board<Square> board) {
        List<Position> newFirePositions = new ArrayList<>();
        if(board.getStepNumber() <= 1){
            return newFirePositions;
        }
        // Fire propagates every 2 turns
        if (age % 2 != 0) {
            return newFirePositions; // No propagation this turn
        }

        // Generate adjacent positions
        List<Position> adjacentPositions = PositionUtil.generateAdjacentPositions(position, board);
        
        for (Position p : adjacentPositions) {
            // Skip if position does not exist
            if (!board.doesPositionExist(p)) {
                continue;
            }

            // Skip if the position contains a Mountain
            if (board.doesSquareContainEntity(p, Mountain.class)) {
                continue;
            }

            // Skip if the position already contains a Fire
            if (board.doesSquareContainEntity(p, Fire.class)) {
                continue;
            }

            // Skip if the position contains a Cloud (if clouds prevent fire spread)
            if (board.doesSquareContainEntity(p, Road.class)) {
                continue;
            }
            if (board.doesSquareContainEntity(p, Rockery.class)) {
                Optional<Entity> e = board.getStates(p).getEntities().stream().findFirst();
                if(e != null && e.isPresent() && e.get() instanceof Rockery){
                    Rockery rockery = (Rockery) e.get();
                    if(rockery.getBurn() < 4){
                        rockery.incrementBurn();
                        continue;
                    }
                    
                }
            }

            // Add new Fire to the board
            board.addEntityAtSquare(new Fire(p), p);
            newFirePositions.add(p); // Keep track of new fires for updates
        }

        return newFirePositions;
    }

    @Override
    public void setPosition(Position p) {
        this.position = p;
    }

    @Override
    public Position getPosition() {
        return this.position;
    }

    @Override
    public Color getViewColor() {
        return this.viewColor;
    }

    @Override
    public int getAge() {
        return this.age;
    }

    @Override
    public void incrementAge() {
        this.age++;
    }

    @Override
    public void setAge(int age) {
        this.age = age;
    }

    @Override
    public int getPriority() {
        return this.priority;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) return true; // Check if same object
        if (obj == null || getClass() != obj.getClass()) return false; // Check for null and class match
        Fire fire = (Fire) obj; // Cast to Fire
        return age == fire.age && position.equals(fire.position); // Compare age and position
    }
    
    
    @Override
    public int hashCode() {
        return Objects.hash(position, age);
    }

}
