package model.firefighterscenario;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;

import javafx.scene.paint.Color;
import model.Board;
import model.EmptySquare;
import model.Entity;
import model.Square;
import util.Position;
import util.PositionUtil;



public class FireFighter implements Entity {
    private int age;
    private Position position;
    private final Color viewColor = Color.BLUE;
    private final int priority = 1;
    protected List<Position> lastThreePosition;

    public FireFighter(Position position, Board<Square> b) {
        this.position = position;
        this.age = 0;
        this.lastThreePosition = new ArrayList<Position>();
    }

    public FireFighter(Position position, Board<Square> b, int age) {
        this.position = position;
        this.age = age;
        this.lastThreePosition = new ArrayList<Position>();
    }

    public List<Position> nextTurn(Board<Square> b) {
        List<Position> positions = new ArrayList<>();
    
        // Generate adjacent positions, excluding mountains
        List<Position> adjacentPositions = PositionUtil.generateAdjacentPositions(position, b);
        adjacentPositions.removeIf(p -> b.doesSquareContainEntity(p, Mountain.class));
    
        // Check if there is fire in any adjacent positions
        boolean hasFire = adjacentPositions.stream()
                .anyMatch(p -> b.doesSquareContainEntity(p, Fire.class));
    
        boolean isStuck = isStuck();
    
        if (hasFire) {
            // Extinguish fires in adjacent positions
            positions.addAll(extinguish(adjacentPositions, b));
            if (isStuck) {
                // If stuck, attempt to move and then extinguish fires
                positions.addAll(moveAndExtinguish(b));
            }
        } else {
            // No fire adjacent; move and attempt to extinguish fires
            positions.addAll(moveAndExtinguish(b));
        }
    
        // Update last three positions
        updateLastPositions();
    
        return positions;
    }
    
    protected boolean isStuck() {
        if (lastThreePosition.size() < 3) {
            return false;
        }
        Position first = lastThreePosition.get(lastThreePosition.size() - 1);
        return lastThreePosition.stream().allMatch(pos -> pos.equals(first));
    }
    
    private void updateLastPositions() {
        if (lastThreePosition.size() >= 3) {
            lastThreePosition.remove(0);
        }
        lastThreePosition.add(this.position);
    }
    

    protected List<Position> extinguish(List<Position> adjacentPositions, Board<Square> b) {
        List<Position> extinguishedPositions = new ArrayList<>();
        for (Position p : adjacentPositions) {
            if (b.doesSquareContainEntity(p, Fire.class)) {
                if(b.doesSquareContainEntity(p, Rockery.class)){
                    Rockery rockery = (Rockery) b.getStates(p).getEntities().stream()
                    .filter(Rockery.class::isInstance)
                    .findFirst().get();
                    rockery.resetBurn();
                }
                b.getStates(p).getEntities().removeIf(element -> element instanceof Fire);
                // Update age for EmptyEntity if needed
                b.getStates(p).getEntities().forEach(e -> {
                    if (e instanceof EmptySquare) {
                        e.setAge(b.stepNumber() + 1);
                    }
                });
                extinguishedPositions.add(p); // Add position where fire was extinguished
            }
        }
        return extinguishedPositions;
    }
    

    protected List<Position> moveAndExtinguish(Board<Square> b) {
        List<Position> positions = new ArrayList<>();
    
        // Find the nearest fire
        Position nearestFirePos = b.getNearestEntity(position, Fire.class);
        if (nearestFirePos != null) {
            // Get the next position towards the fire
            Position nextPos = getNextPositionTowards(position, nearestFirePos, b);
            if (nextPos != null && !nextPos.equals(position)) {
                // Move the firefighter
                b.clearCaseFrom(this, position); // Clear old position
                positions.add(new Position(position.x(), position.y())); // Add old position
                this.position = nextPos;
                b.addEntityAtSquare(this, nextPos); // Add to new position
                positions.add(nextPos); // Add new position
    
                // After moving, attempt to extinguish fires adjacent to new position
                List<Position> newAdjacentPositions = PositionUtil.generateAdjacentPositions(nextPos, b);
                newAdjacentPositions.removeIf(p -> b.doesSquareContainEntity(p, Mountain.class));
                positions.addAll(extinguish(newAdjacentPositions, b));
            }
        }
        return positions;
    }
    
    protected Position getNextPositionTowards(Position currentPos, Position targetPos, Board<Square> b) {
        // Generate adjacent positions
        List<Position> possibleMoves = PositionUtil.generateAllAdjacentPositions(currentPos, b);
    
        // Filter out positions that are not empty or contain obstacles
        possibleMoves.removeIf(p -> b.doesSquareContainEntity(p, Mountain.class));
    
        // If no possible moves, return null
        if (possibleMoves.isEmpty()) {
            return null;
        }
    
        // Calculate the current distance to the target
        int currentDistance = PositionUtil.getManhattanDistance(currentPos, targetPos);
    
        // Initialize variables to find the best moves
        int minDistance = Integer.MAX_VALUE;
        List<Position> bestMoves = new ArrayList<>();
    
        for (Position move : possibleMoves) {
            int distance = PositionUtil.getManhattanDistance(move, targetPos);
    
            // Skip positions occupied by other firefighters
            if (b.doesSquareContainEntity(move, FireFighter.class)) {
                continue;
            }
    
            // Find positions that minimize the distance
            if (distance < minDistance) {
                minDistance = distance;
                bestMoves.clear();
                bestMoves.add(move);
            } else if (distance == minDistance) {
                bestMoves.add(move);
            }
        }
    
        // If no better move is found, consider moves that maintain the same distance
        if (bestMoves.isEmpty()) {
            minDistance = currentDistance;
            for (Position move : possibleMoves) {
                int distance = PositionUtil.getManhattanDistance(move, targetPos);
                if (distance == minDistance) {
                    bestMoves.add(move);
                }
            }
        }
    
        // If still no move is found, stay in the current position
        if (bestMoves.isEmpty()) {
            return currentPos;
        }
    
        // Select a move from the best moves (e.g., randomly or based on additional criteria)
        Random r = new Random();
        
        Position nextMove = bestMoves.get(r.nextInt(bestMoves.size()));
    
        return nextMove;
    }
    

    @Override
    public void setPosition(Position p) {
        this.position = p;
    }

    @Override
    public Position getPosition() {
        return this.position;
    }

    public Color getViewColor() {
        return this.viewColor;
    }

    @Override
    public int getAge() {
        return this.age;
    }

    @Override
    public void incrementAge() {
        this.age = age + 1;
    }

    @Override
    public void setAge(int age) {
        this.age = age;
    }
    public int getPriority(){ return this.priority;}

}
