package model.firefighterscenario;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Random;

import app.SimulatorApplication;
import model.Board;
import model.Entity;
import model.EntityFactory;
import model.EntityScenario;
import model.EntitySpawner;
import model.Square;
import util.Matrix;
import util.PathGenerator;
import util.Position;
import util.PositionUtil;

public class FireFighterScenario extends EntityScenario implements Board<Square> {

  private Matrix<Square> matrix;
  private int step;
  private int turnsToSpawnAirTanker;

  private Map<EntityFactory, Integer> initialMap;

  public FireFighterScenario(int columns, int rows) {
    this.matrix = new Matrix<Square>(columns, rows);
    initScenario(matrix);
    this.turnsToSpawnAirTanker = SimulatorApplication.TURNS_FOR_SPAWNING_AIRTANKER;
    this.step = 0;
  }

  public void placeInitialEntities(Map<EntityFactory, Integer> entityCounts) {
    EntitySpawner spawner = new EntitySpawner(this);
    spawner.spawnEntities(entityCounts);
    generateRoads();
    this.initialMap = entityCounts;
  }

  public Square getStates(Position position) {
    if (position.x() > matrix.size() || position.y() > matrix.size()) {
      throw new IllegalArgumentException(
          "The position x:" + position.x() + " y:" + position.y() + " is out of the board.");
    }
    return matrix.get(position.x(), position.y());
  }

  public void setSquare(Square square) {
    Position position = square.getPosition();
    if (!(getStates(position).isEmpty())) {
      return;
    }
    if (doesPositionExist(position)) {
      matrix.set(position.x(), position.y(), square);
    }
  }

  public void setSquare(Square square, boolean replaceStates) {
    Position position = square.getPosition();
    if (!(getStates(position).isEmpty()) && !replaceStates) {
      return;
    }
    matrix.set(position.x(), position.y(), square);
  }

  public void addEntityAtSquare(Entity entity, Position position) {
    if (doesPositionExist(position)) {
      matrix.get(position.x(), position.y()).addEntity(entity);
    }
  }

  public int rowCount() {
    return matrix.getRows();
  }

  public int columnCount() {
    return matrix.getColumns();
  }

  @Override
  public void clearCaseFrom(Entity entity, Position position) {
    matrix.get(position.x(), position.y()).getEntities().removeIf(element -> element.equals(entity));
  }

  public List<Position> updateToNextGeneration() {
    ArrayList<Position> changedPositions = new ArrayList<>();
    Iterator<Square> iterator = matrix.iterator();

    while (iterator.hasNext()) {
      Square s = iterator.next();
      if (s.isEmpty())
        continue;
      if (s.getMaxAge() == 0) {
        s.incrementAllAges();
        continue;
      }
      if (s.getMaxAge() == step + 1) {
        continue;
      }
      List<Entity> entities = new ArrayList<>(s.getEntities());
      for (Entity e : entities) {
        if (e.getAge() >= stepNumber() - 1) {
          continue;
        }
        e.incrementAge();
        changedPositions.addAll(e.nextTurn(this));
      }
    }

    // Increment the step counter
    this.step = this.step + 1;

    // Check if it's time to spawn an AirTanker
    if (this.step % this.turnsToSpawnAirTanker == 0) {
      System.out.println("apparation");
      // Spawn an AirTanker at a random edge position
      spawnAirTanker(changedPositions);
    }

    return changedPositions;
  }

  // Helper method to spawn an AirTanker
  private void spawnAirTanker(List<Position> changedPositions) {
    Random rand = new Random();
    int edge = rand.nextInt(4); // 0: top, 1: bottom, 2: left, 3: right
    Position position = null;

    if (edge == 0) { // Top edge (x == 0)
      int y = rand.nextInt(columnCount() - 1);
      position = new Position(0, y);
    } else if (edge == 1) { // Bottom edge (x == rowCount() - 1)
      int y = rand.nextInt(columnCount());
      position = new Position(rowCount() - 1, y);
    } else if (edge == 2) { // Left edge (y == 0)
      int x = rand.nextInt(rowCount() - 1);
      position = new Position(x, 0);
    } else if (edge == 3) { // Right edge (y == columnCount() - 1)
      int x = rand.nextInt(rowCount() - 1);
      position = new Position(x, columnCount() - 1);
    } else {
      // This else block is technically not necessary because edge will always be
      // between 0 and 3
      throw new IllegalStateException("Unexpected edge value: " + edge);
    }

    // Create a new AirTanker
    AirTanker airTanker = new AirTanker(position, this, getStepNumber());
    // Add the AirTanker to the board
    addEntityAtSquare(airTanker, position);

    // Record the changed position
    changedPositions.add(position);
  }

  public Position getNearestEntity(Position fromPos, Class<?> entityType) {
    int rows = matrix.getRows();
    int cols = matrix.getColumns();
    Position nearestPosition = fromPos;

    // Définir la distance maximale possible
    int maxDistance = rows + cols;
    // Parcourir les distances croissantes à partir de 1
    for (int distance = 1; distance < maxDistance; distance++) {
      List<Position> positionsAtDistance = PositionUtil.getPositionsAtManhattanDistance(fromPos, distance, rows, cols);

      for (Position currentPos : positionsAtDistance) {
        Square currentSquare = matrix.get(currentPos.x(), currentPos.y());
        for (Entity currentEntity : currentSquare.getEntities()) {
          if (entityType.isInstance(currentEntity)) {
            // Dès qu'une entité est trouvée à cette distance, elle est la plus proche
            // possible
            return currentPos;
          }
        }
      }
    }

    return nearestPosition; // Retourne null si aucune entité n'est trouvée
  }

  public void reset() {
    step = 0;
    matrix.clear();
    initScenario(matrix);
    placeInitialEntities(initialMap);
  }

  public int stepNumber() {
    return this.step;
  }

  @Override
  public boolean doesPositionExist(Position position) {
    return matrix.validateIndex(position);
  }

  @Override
  public int getStepNumber() {
    return step;
  }

  @Override
  public boolean doesSquareContainEntity(Position squarePos, Class<?> entityType) {
    return getStates(squarePos).getEntities().stream().anyMatch(entityType::isInstance);
  }

  @Override
  public boolean isPositionEmpty(Position position) {
    return getStates(position).isEmpty();
  }

  @Override
  public boolean isPositionFree(Position position, int priority) {
    List<Entity> entities = matrix.get(position.x(), position.y()).getEntities();
    for (Entity e : entities) {
      if (e.getPriority() == priority) {
        return false;
      }
    }
    return true;
  }

  private void generateRoads() {
    if(columnCount() < 10 || rowCount() < 10){
      return;
    }
    Random random = new Random();

    // Get board dimensions
    int rowCount = rowCount();      // Number of rows (vertical axis)
    int columnCount = columnCount(); // Number of columns (horizontal axis)

    // Decide randomly whether to set x or y to 0
    boolean setXToZero = random.nextBoolean();

    int x = 0;
    int y = 0;

    if (setXToZero) {
        // x is set to 0, y is random within column bounds
        x = 0;
        y = random.nextInt(columnCount);
    } else {
        // y is set to 0, x is random within row bounds
        x = random.nextInt(rowCount);
        y = 0;
    }

    Position startPosition = new Position(x, y);
    PathGenerator pathGenerator = new PathGenerator(this, startPosition, 4, columnCount());
    // Call generateEntitiesInLine to place the roads
    List<Position> snake = pathGenerator.generate();
    for(Position p : snake){
      List<Entity> entitiesAtSquare = List.copyOf(getStates(p).getEntities());
      for(Entity e: entitiesAtSquare){
        if(e instanceof Mountain || e instanceof Rockery || e instanceof Fire){
          clearCaseFrom(e, e.getPosition());
        }
      }
      addEntityAtSquare(new Road(p), p);
    }
}

}
