package model.firefighterscenario;

import java.util.ArrayList;
import java.util.List;

import javafx.scene.paint.Color;
import model.Board;
import model.Square;
import util.Position;
import util.PositionUtil;

public class MotorizedFireFighter extends FireFighter {
    private final Color viewColor = Color.CYAN;

    public MotorizedFireFighter(Position position, Board<Square> b) {
        super(position, b);
    }

    @Override
    public List<Position> nextTurn(Board<Square> b) {
        List<Position> positions = new ArrayList<>();

        // Generate adjacent positions excluding mountains
        List<Position> adjacentPositions = PositionUtil.generateAdjacentPositions(getPosition(), b);
        adjacentPositions.removeIf(p -> b.doesSquareContainEntity(p, Mountain.class));

        // Check if there is fire in any adjacent positions
        boolean hasFire = adjacentPositions.stream()
                .anyMatch(p -> b.doesSquareContainEntity(p, Fire.class));

        boolean isStuck = isStuck();

        if (hasFire) {
            // Extinguish fires in adjacent positions
            positions.addAll(extinguish(adjacentPositions, b));

            if (isStuck) {
                // If stuck, attempt to move and extinguish after moving
                positions.addAll(moveAndExtinguish(b));
            }
        } else {
            // No fire adjacent; move and attempt to extinguish fires
            positions.addAll(moveAndExtinguish(b));
        }

        // Update last positions for stuck detection
        updateLastPositions();

        return positions;
    }

    @Override
    protected List<Position> moveAndExtinguish(Board<Square> b) {
        List<Position> positions = new ArrayList<>();

        // Find the nearest fire
        Position nearestFirePos = b.getNearestEntity(getPosition(), Fire.class);
        if (nearestFirePos != null) {
            // Get the next position after moving up to two steps towards the fire
            Position nextPos = getNextPositionTowards(getPosition(), nearestFirePos, b, 2);

            if (nextPos != null && !nextPos.equals(getPosition())) {
                // Move the firefighter
                b.clearCaseFrom(this, getPosition()); // Clear old position
                positions.add(new Position(getPosition().x(), getPosition().y())); // Add old position
                setPosition(nextPos);
                b.addEntityAtSquare(this, nextPos); // Add to new position
                positions.add(nextPos); // Add new position

                // After moving, attempt to extinguish fires adjacent to new position
                List<Position> newAdjacentPositions = PositionUtil.generateAdjacentPositions(nextPos, b);
                newAdjacentPositions.removeIf(p -> b.doesSquareContainEntity(p, Mountain.class));
                positions.addAll(extinguish(newAdjacentPositions, b));
            }
        }

        return positions;
    }

    // Overloaded method to handle multiple steps
    protected Position getNextPositionTowards(Position currentPos, Position targetPos, Board<Square> b, int steps) {
        Position nextPos = currentPos;

        for (int i = 0; i < steps; i++) {
            Position stepPos = super.getNextPositionTowards(nextPos, targetPos, b);

            if (stepPos == null || stepPos.equals(nextPos)) {
                // Can't move further
                break;
            }

            nextPos = stepPos;
        }

        return nextPos;
    }

    // Override to ensure correct color
    @Override
    public Color getViewColor() {
        return this.viewColor;
    }

    // Ensure the last positions are updated correctly
    protected void updateLastPositions() {
        if (lastThreePosition.size() >= 3) {
            lastThreePosition.remove(0);
        }
        lastThreePosition.add(this.getPosition());
    }

}