package util;

import java.util.Arrays;
import java.util.List;
import java.util.Random;
import java.util.Set;
import java.util.stream.Collectors;

import model.Board;
import model.Square;

public class DirectionUtils {
    
    /**
     * Obtient l'incrément sur l'axe X pour une direction donnée.
     *
     * @param direction La direction.
     * @return L'incrément sur l'axe X.
     */
    public static int getXIncrement(Direction direction) {
        switch (direction) {
            case NORTH:
                return -1;
            case SOUTH:
                return 1;
            default:
                return 0;
        }
    }

    /**
     * Obtient l'incrément sur l'axe Y pour une direction donnée.
     *
     * @param direction La direction.
     * @return L'incrément sur l'axe Y.
     */
    public static int getYIncrement(Direction direction) {
        switch (direction) {
            case EAST:
                return 1;
            case WEST:
                return -1;
            default:
                return 0;
        }
    }

    /**
     * Obtient la direction opposée.
     *
     * @param direction La direction actuelle.
     * @return La direction opposée.
     */
    public static Direction getOpposite(Direction direction) {
        switch (direction) {
            case NORTH:
                return Direction.SOUTH;
            case SOUTH:
                return Direction.NORTH;
            case EAST:
                return Direction.WEST;
            case WEST:
                return Direction.EAST;
            default:
                throw new IllegalArgumentException("Direction non supportée : " + direction);
        }
    }

    /**
     * Choisit une direction aléatoire parmi toutes les directions disponibles.
     *
     * @param random Instance de Random.
     * @return Une direction aléatoire.
     */
    public static Direction getRandomDirection() {
        Random r = new Random();
        Direction[] directions = Direction.values();
        return directions[r.nextInt(directions.length)];
    }

    /**
     * Obtient les directions valides en fonction des exclusions et de la position actuelle.
     *
     * @param currentDirection              La direction actuelle.
     * @param permanentlyExcludedDirections Les directions définitivement exclues.
     * @param temporarilyExcludedDirections Les directions temporairement exclues.
     * @param board                         Le plateau de jeu.
     * @param currentPosition               La position actuelle.
     * @return Une liste de directions valides.
     */
    public static List<Direction> getValidDirections(
            Direction currentDirection,
            Set<Direction> permanentlyExcludedDirections,
            Set<Direction> temporarilyExcludedDirections,
            Board<Square> board,
            Position currentPosition
    ) {
        return Arrays.stream(Direction.values())
                .filter(dir -> dir != currentDirection)
                .filter(dir -> dir != getOpposite(currentDirection))
                .filter(dir -> !permanentlyExcludedDirections.contains(dir))
                .filter(dir -> !temporarilyExcludedDirections.contains(dir))
                .filter(dir -> {
                    Position nextPos = currentPosition.move(dir);
                    return board.doesPositionExist(nextPos);
                })
                .collect(Collectors.toList());
    }
}
