package util;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import model.Board;
import model.Square;

public class PositionUtil {
    /**
     * Calcule et retourne une liste de positions situées à une distance de
     * Manhattan spécifique à partir d'une position donnée.
     * La distance de Manhattan entre deux points (x1, y1) et (x2, y2) est donnée
     * par |x1 - x2| + |y1 - y2|.
     * Cette méthode génère toutes les positions possibles à cette distance dans une
     * grille de taille spécifiée.
     *
     * @param fromPos  la position de départ à partir de laquelle la distance est
     *                 calculée.
     * @param distance la distance de Manhattan à utiliser pour trouver les
     *                 positions.
     * @param rows     le nombre de lignes dans la grille.
     * @param cols     le nombre de colonnes dans la grille.
     * @return une liste de positions à la distance de Manhattan spécifiée de la
     *         position initiale, qui se trouvent également dans les limites de la
     *         grille.
     */
    public static List<Position> getPositionsAtManhattanDistance(Position fromPos, int distance, int rows, int cols) {
        List<Position> positions = new ArrayList<>();
        int x0 = fromPos.x();
        int y0 = fromPos.y();

        // Générer toutes les positions à une distance de Manhattan donnée
        for (int dx = -distance; dx <= distance; dx++) {
            int dy = distance - Math.abs(dx);

            int[] dyOptions = { dy, -dy };
            for (int deltaY : dyOptions) {
                int x = x0 + dx;
                int y = y0 + deltaY;

                // Vérifier si la position est dans les limites de la matrice
                if (x >= 0 && x < rows && y >= 0 && y < cols) {
                    positions.add(new Position(x, y));
                }
            }
        }

        return positions;
    }

    /**
     * Génère une liste de positions adjacentes à une position donnée, en vérifiant
     * si chaque position est valide dans le contexte du jeu.
     *
     * @param position la position de départ pour laquelle générer les positions
     *                 adjacentes.
     * @param board    l'objet représentant le plateau de jeu qui permet de vérifier
     *                 si une position existe.
     * @return une liste des positions adjacentes valides.
     */
    public static List<Position> generateAdjacentPositions(Position position, Board<Square> board) {
        int x = position.x();
        int y = position.y();

        return Stream.of(
                new Position(x, y + 1),
                new Position(x + 1, y),
                new Position(x, y - 1),
                new Position(x - 1, y))
                .filter(p -> board.doesPositionExist(p))
                .collect(Collectors.toList());
    }

    public static List<Position> generateAdjacentPositions(Position position, Board<Square> board, int distance) {

        List<Position> positions = new ArrayList<Position>();
        for(int i = 0; i < 4; i++){

        }
        return positions;
    }


    // Méthode pour générer toutes les positions adjacentes (y compris les
    // diagonales)
    public static List<Position> generateAllAdjacentPositions(Position position, Board<Square> board) {
        int x = position.x();
        int y = position.y();

        List<Position> positions = new ArrayList<>();

        // Liste des 8 déplacements possibles
        int[][] deltas = {
                { 1, 0 }, // x+1, y
                { -1, 0 }, // x-1, y
                { 0, 1 }, // x, y+1
                { 0, -1 }, // x, y-1
                { 1, 1 }, // x+1, y+1
                { -1, -1 }, // x-1, y-1
                { 1, -1 }, // x+1, y-1
                { -1, 1 } // x-1, y+1
        };

        for (int[] delta : deltas) {
            int newX = x + delta[0];
            int newY = y + delta[1];
            Position p = new Position(newX, newY);
            if (board.doesPositionExist(p)) {
                positions.add(p);
            }
        }

        return positions;
    }

    public static int getManhattanDistance(Position p1, Position p2) {
        return Math.abs(p1.x() - p2.x()) + Math.abs(p1.y() - p2.y());
    }

        /**
     * Generates all positions within a specified radius from a center position.
     * It uses Manhattan distance by default.
     *
     * @param center The center position.
     * @param radius The radius (distance) from the center position.
     * @param board  The board to consider boundaries.
     * @return A list of positions within the radius.
     */
    public static List<Position> getPositionsInRadius(Position center, int radius, Board<Square> board) {
        List<Position> positions = new ArrayList<>();

        int startX = center.x() - radius;
        int endX = center.x() + radius;
        int startY = center.y() - radius;
        int endY = center.y() + radius;

        for (int x = startX; x <= endX; x++) {
            for (int y = startY; y <= endY; y++) {
                Position pos = new Position(x, y);
                if (board.doesPositionExist(pos)) {
                    // Calculate Manhattan distance from the center
                    int distance = Math.abs(center.x() - x) + Math.abs(center.y() - y);
                    if (distance <= radius) {
                        positions.add(pos);
                    }
                }
            }
        }

        return positions;
    }

}